#include <float.h>
#include <math.h>
#include <assert.h>

#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnTransform.h>
#include <maya/MGlobal.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MFnAnimCurve.h>
#include <maya/MAnimControl.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MItDependencyGraph.h>

#include <MDt.h>
#include <MDtExt.h>

#include "animation.h"
#include "querymel.h"
#include "global.h"

/* animKey member functions */

animKey::animKey(float newTime)
{
    isKeyFrame  = false;
    time        = newTime;
    dirty       = true;
    nodeIndex   = -1;
    RwMatrixSetIdentity(&matrix);
}

animKey::animKey(float newTime, bool keyFrame)
{
    isKeyFrame  = keyFrame;
    time        = newTime;
    dirty       = true;
    nodeIndex   = -1;
    RwMatrixSetIdentity(&matrix);
}

animKey::animKey(float newTime, RwMatrix *newMatrix, bool keyFrame)
{
    isKeyFrame  = keyFrame;
    time        = newTime;
    dirty       = false;
    nodeIndex   = -1;

    if (newMatrix != NULL)
    {
        RwMatrixCopy(&matrix, newMatrix);
        RpAnimQuatConvertFromMatrix(&quat, &matrix);
    }
    else
    {
        RwMatrixSetIdentity(&matrix);
    }
}

bool animKey::operator==(const animKey &other) const
{
    if (other.time != time)
    {
        return false;
    }

    if (other.nodeIndex != nodeIndex)
    {
        return false;
    }

    return true;
}

bool animKey::operator<(const animKey &other) const
{
    if (time < other.time)
    {
        return true;
    }

    if (nodeIndex < other.nodeIndex)
    {
        return true;
    }

    return false;
}

animKey & animKey::operator*=(animKey &b)
{
    RwMatrix dstMatrix;

    RwMatrixMultiply(&dstMatrix, &matrix, &b.matrix);
    
    RwMatrixCopy(&matrix, &dstMatrix);

    RpAnimQuatConvertFromMatrix(&quat, &matrix);

    if (b.isKeyFrame == true)
    {
        isKeyFrame = true;
    }

    return *this;
}

/* animKeyList member functions */

animKeyList & animKeyList::operator*=(animKeyList &b)
{
    animKeyIt i, j;

    assert(keys.size() == b.keys.size());

    for (i = keys.begin(), j = b.keys.begin(); i != keys.end(); i++, j++)
    {
        (*i) *= (*j);
    }

    return *this;
}

float animKeyList::getKeyRange()
{
    animKeyIt min, max;

    min = min_element(keys.begin(), keys.end());
    max = max_element(keys.begin(), keys.end());
    
    return RwRealAbs(max->time - min->time);
}

void animKeyList::copyKeyFrames(animKeyList &src, animKeyList &dst, float offset, bool reverse)
{
    MStatus     status;	
    float       oldTime, newTime, range;
    animKeyIt   i, start, end;

    if (reverse)
    {
        range = src.getKeyRange();
    }

    start = src.keys.begin();
    end = src.keys.end();

    for (i = start; i != end; i++)
    {
        oldTime = i->time - start->time;
        
        if (reverse)
        {
            newTime = offset + (range - oldTime);
        }
        else
        {
            newTime = offset + oldTime;
        }

        dst.keys.insert(newTime);
    }
}

void animKeyList::dumpKeyFrames()
{
    animKeyIt i;

    printf("\n\n\n");

    for (i = keys.begin(); i != keys.end(); i++)
    {
        printf("Time = %f Keyframe = %s\n", i->time, i->isKeyFrame ? "TRUE" : "FALSE");
    }
}

bool animKeyList::hasRotationAnimation()
{
    animKeyIt i;

    for (i = keys.begin(); i != keys.end(); i++)
    {
        if (i->isKeyFrame == true)
        {
            if ((i->quat.real != keys.begin()->quat.real) ||
                (i->quat.imag.x != keys.begin()->quat.imag.x) ||
                (i->quat.imag.y != keys.begin()->quat.imag.y) ||
                (i->quat.imag.z != keys.begin()->quat.imag.z))
            {
                return true;
            }
        }
    }
    return false;
}

bool animKeyList::hasTranslationAnimation()
{
    animKeyIt i;

    for (i = keys.begin(); i != keys.end(); i++)
    {
        if (i->isKeyFrame == true)
        {
            if ((i->matrix.pos.x != keys.begin()->matrix.pos.x) ||
                (i->matrix.pos.y != keys.begin()->matrix.pos.y) ||
                (i->matrix.pos.z != keys.begin()->matrix.pos.z))
            {
                return true;
            }
        }
    }
    return false;
}

bool animKeyList::checkKeyValueError(float startTime, float endTime,
                                     float startValue, float endValue,
                                     float interpEndValue, float errorTolerance,
                                     float keyrange)
{
    if (RwRealAbs(interpEndValue - endValue) <= RwRealAbs((errorTolerance / 100.0f) * keyrange))
    {
        return true;
    }
    else
    {
        return false;
    }
}

bool animKeyList::checkKeyValueErrors(animKeyIt startKey, animKeyIt middleKey,
                           animKeyIt endKey, float errorTolerance)
{
    float       slerpValue;
    RtSlerp     *slerpCache;
    RwMatrix    interpolatedKey;
    RpQuat      interpolatedQuat;

    /* Calculate the interpolated matrix at the middle key time */
    slerpValue = ((middleKey->time - startKey->time) / (endKey->time - startKey->time));
    
    slerpCache = RtSlerpCreate(rtSLERPREFNONE);

    if (slerpCache != NULL)
    {
        if (RtSlerpInitialize(slerpCache, &startKey->matrix, &endKey->matrix))
        {
            RtSlerpGetMatrix(slerpCache, &interpolatedKey, slerpValue);
            RpAnimQuatConvertFromMatrix(&interpolatedQuat, &interpolatedKey);
        }
        RtSlerpDestroy(slerpCache);
    }

    /* Check position errors */
    if (posRange.x > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->matrix.pos.x,
                                middleKey->matrix.pos.x, interpolatedKey.pos.x,
                                errorTolerance, posRange.x))
        {
            return false;
        }
    }
    
    if (posRange.y > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->matrix.pos.y,
                                middleKey->matrix.pos.y, interpolatedKey.pos.y,
                                errorTolerance, posRange.y))
        {
            return false;
        }
    }
    
    if (posRange.z > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->matrix.pos.z,
                                middleKey->matrix.pos.z, interpolatedKey.pos.z,
                                errorTolerance, posRange.z))
        {
            return false;
        }
    }

    /* Check rotation errors */
    if (quatRange.real > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->quat.real,
                                middleKey->quat.real, interpolatedQuat.real,
                                errorTolerance, quatRange.real))
        {
            return false;
        }
    }

    if (quatRange.imag.x > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->quat.imag.x,
                                middleKey->quat.imag.x, interpolatedQuat.imag.x,
                                errorTolerance, quatRange.imag.x))
        {
            return false;
        }
    }
    
    if (quatRange.imag.y > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->quat.imag.y,
                                middleKey->quat.imag.y, interpolatedQuat.imag.y,
                                errorTolerance, quatRange.imag.y))
        {
            return false;
        }
    }
    
    if (quatRange.imag.z > SMALL_FLOAT)
    {
        if (!checkKeyValueError(startKey->time, middleKey->time, startKey->quat.imag.z,
                                middleKey->quat.imag.z, interpolatedQuat.imag.z,
                                errorTolerance, quatRange.imag.z))
        {
            return false;
        }
    }

    /* All the errors were within our threshold so return true. */
    return true;
}

void animKeyList::checkKeyFrameRangeError(animKeyIt startKey, animKeyIt endKey,
                                          int rangeKeyCount, float errorTolerance)
{
    int         i, middleKeyCount;
    animKeyIt   middleKey;

    middleKeyCount = rangeKeyCount >> 1;

    if (middleKeyCount > 0)
    {
        middleKey = startKey;
        for (i = 0; i < middleKeyCount; i++)
        {
            middleKey++;
        }

        if (!checkKeyValueErrors(startKey, middleKey, endKey, errorTolerance))
        {
            middleKey->isKeyFrame = true;
        }

        checkKeyFrameRangeError(startKey, middleKey, middleKeyCount, errorTolerance);
        checkKeyFrameRangeError(middleKey, endKey, (rangeKeyCount - middleKeyCount),
                                errorTolerance);
    }
}

inline void animKeyList::rangeCheck(float &min, float &max, float value)
{
    if (value < min)
    {
        min = value;
    }
    if (value > max)
    {
        max = value;
    }
}

void animKeyList::computeKeyFrameRange()
{
    animKeyIt   i;
    RwV3d       posMin, posMax;
    RpQuat      quatMin, quatMax;

    posMin.x = posMin.y = posMin.z = FLT_MAX;
    posMax.x = posMax.y = posMax.z = -FLT_MAX;

    quatMin.imag.x = quatMin.imag.y = quatMin.imag.z = quatMin.real = FLT_MAX;
    quatMax.imag.x = quatMax.imag.y = quatMax.imag.z = quatMax.real = -FLT_MAX;

    for (i = keys.begin(); i != keys.end(); i++)
    {
        rangeCheck(posMin.x, posMax.x, i->matrix.pos.x);
        rangeCheck(posMin.y, posMax.y, i->matrix.pos.y);
        rangeCheck(posMin.z, posMax.z, i->matrix.pos.z);

        rangeCheck(quatMin.imag.x, quatMax.imag.x, i->quat.imag.x);
        rangeCheck(quatMin.imag.y, quatMax.imag.y, i->quat.imag.y);
        rangeCheck(quatMin.imag.z, quatMax.imag.z, i->quat.imag.z);
        rangeCheck(quatMin.real, quatMax.real, i->quat.real);
    }

    posRange.x = RwRealAbs(posMax.x - posMin.x);
    posRange.y = RwRealAbs(posMax.y - posMin.y);
    posRange.z = RwRealAbs(posMax.z - posMin.z);

    quatRange.imag.x = RwRealAbs(quatMax.imag.x - quatMin.imag.x);
    quatRange.imag.y = RwRealAbs(quatMax.imag.y - quatMin.imag.y);
    quatRange.imag.z = RwRealAbs(quatMax.imag.z - quatMin.imag.z);
    quatRange.real  = RwRealAbs(quatMax.real - quatMin.real);
}

void animKeyList::checkKeyFrameErrors(float errorTolerance)
{
    animKeyIt   key, start, end;
    int         interpKeyCount = 0;

    /* First figure out the range of values across our keyframes.*/ 
    computeKeyFrameRange();

    for (start = end = keys.begin();; start = end)
    {
        /* Find two marked keyframes */

        assert(start->isKeyFrame == true);

        interpKeyCount = 0;
        while ((end == start) || (end->isKeyFrame == false))
        {
            end++;
            if (end == keys.end())
            {
                return;
            }
            interpKeyCount++;
        }

        /* Check the error at intermediate keys */
        checkKeyFrameRangeError(start, end, interpKeyCount, errorTolerance);
    }
}

void animKeyList::computeAnimation(int mdtIndex, MDagPath dagPath)
{
    MObject     transformNode;
    bool        ik = false;
    animKeyIt   key, tempKey;
    int         i;

    /* Get the keyframes on the shape itself */
    DtExt_ShapeGetTransform(mdtIndex, transformNode);
    addAnimKeysFromTransform(transformNode);

    /* And add in any extra keys based on IK */
    _ikHandleList *ikhlist = globalData->gIkHandleList;
    while(ikhlist)
    {
        MFnDagNode shapeDagFn(dagPath);

        if ((dagPath == *ikhlist->startJoint || shapeDagFn.isChildOf(ikhlist->startJoint->node())) &&
            (dagPath == *ikhlist->endEffector || shapeDagFn.isParentOf(ikhlist->endEffector->node())))
        {
            MStatus status;

            /* add keys from the handle */
            addAnimKeysFromTransform(*ikhlist->handle);
            
            /* add keys from all parents to this object */
            addKeysFromParents(shapeDagFn);

            /* and keys from the parents of the IK handle */
            MFnDagNode handleDagFn(*ikhlist->handle, &status);
            addKeysFromParents(handleDagFn);

            keys.insert(animKey((float)globalData->m_animStart, true));
            keys.insert(animKey((float)globalData->m_animEnd, true));
            ik = true;
        }
        
        ikhlist = ikhlist->next;
    }

    /* remove any keys outside the set anim range */
    for (key = keys.begin(); key != keys.end();)
    {
        RwReal keyTime = key->time;
        key++;

        if ((keyTime < (float)globalData->m_animStart) ||
            (keyTime > (float)globalData->m_animEnd))
        {
            keys.erase(keyTime);
        }
    }
    
    if (globalData->m_exportRpSkin  ||
        globalData->m_exportRpHAnim ||
        (keys.size() > 0))
    {
        keys.insert(animKey((float)globalData->m_animStart, true));
        keys.insert(animKey((float)globalData->m_animEnd, true));
    }
    
    /*
        If we want to dynamically generate keyframes based on errors
        then we need to sample Maya matrices at every keyframe. We
        later throw away the ones we don't need.
    */
    if (globalData->m_dynamicKeyframeGeneration && (keys.size() > 0))
    {
        for (i = globalData->m_animStart; i < globalData->m_animEnd; i++)
        {
            keys.insert(animKey((float)i, false));
        }
    }
}

void animKeyList::removeRedundantKeys()
{
    animKeyIt key;

    /* Remove any keys that aren't marked as keyframes */
    for (key = keys.begin(); key != keys.end();)
    {
        RwReal keyTime  = key->time;
        bool isKeyFrame = key->isKeyFrame;

        key++;

        if (isKeyFrame == false)
        {
            keys.erase(animKey(keyTime));
        }
    }
}

int animKeyList::countValidKeys()
{
    int         count = 0;
    animKeyIt   key;

    for (key = keys.begin(); key != keys.end(); key++)
    {
        if (key->isKeyFrame == true)
        {
           count++;
        }
    }

    return count;
}
void animKeyList::addAnimKeysFromTransform(MObject transform)
{
    MStatus status;
    MObject anim;

    MItDependencyGraph::Direction direction = MItDependencyGraph::kUpstream;
    MItDependencyGraph::Traversal traversalType = MItDependencyGraph::kBreadthFirst;
    MItDependencyGraph::Level level = MItDependencyGraph::kNodeLevel;

    MFn::Type filter = MFn::kAnimCurve;
    
    MItDependencyGraph dgIter(transform, filter, direction,
        traversalType, level, &status);
    
    for (; !dgIter.isDone(); dgIter.next())
    {  
        anim = dgIter.thisNode(&status);

        addAnimKeysFromAnimCurve(anim, transform);
    }
}

void animKeyList::addKeysFromParents(MFnDagNode &nodeDagFn)
{
    MStatus status;
    MObject parent;
    int     parentCount;

    parentCount = nodeDagFn.parentCount(&status);
    
    if(parentCount > 0)
    {
        parent = nodeDagFn.parent(0, &status);

        addAnimKeysFromTransform(parent);
    
        MFnDagNode parentDagFn(parent);

        addKeysFromParents(parentDagFn);
    }
}

void animKeyList::addAnimKeysFromAnimCurve(MObject anim, MObject transform)
{
    MStatus             status;	
    MFnAnimCurve        animCurve(anim, &status);
    animCurveType       curveType;
    int                 i, numKeys, numNewKeyFrames;
    RwReal              keyTime0, keyTime1, newKeyTime;
    MTime               mTime0, mTime1, newMTime;
    MFnDependencyNode   curveDepNode(anim, &status);
    MString             curveName = curveDepNode.name(&status);
    const char          *cCurveName = curveName.asChar();
    animKeyList         newKeys, animKeysCopy;
    animKeyIt           key;
    RwReal              animCurveRange, animCurveOffset;

    curveType = getAnimCurveType(anim, transform);;

    numKeys = animCurve.numKeyframes(&status);

    /* Set the units on our MTime variables */
    if (numKeys == 0)
    {
        return;
    }

    mTime0.setUnit(MTime::uiUnit());
    mTime1.setUnit(MTime::uiUnit());
    newMTime.setUnit(MTime::uiUnit());

    /* Create keyframes for all the existing points on the curve */
    for (i = 0; i < numKeys; i++)
    {
        keyTime0 = (RwReal) animCurve.time(i, &status).value();
        newKeys.keys.insert(animKey(keyTime0, true));
    }

    /*
        If this is a rotation curve then check if any of the keyframe
        deltas exceed 180 degrees. If they do then introduce extra
        keyframes as necessary. 
    */
    if (curveType == rotationAnimCurve)
    {
        animKeysCopy = newKeys;

        for (key = animKeysCopy.keys.begin(); key != animKeysCopy.keys.end();)
        {
            double rotation0, rotation1;

            keyTime0 = key->time;
            key++;
            keyTime1 = key->time;

            mTime0.setValue(keyTime0);
            mTime1.setValue(keyTime1);

            rotation0 = animCurve.evaluate(mTime0, &status);
            rotation1 = animCurve.evaluate(mTime1, &status);

            if (RwRealAbs(rotation1 - rotation0) > (0.9 * rwPI))
            {
                numNewKeyFrames = (int)(RwRealAbs(rotation1 - rotation0) / (rwPI * 0.9));
                for (i = 0; i < numNewKeyFrames; i++)
                {
                    newKeyTime = keyTime0 +
                        ((keyTime1 - keyTime0) * ((float)(i + 1) / (numNewKeyFrames + 1)));
                    newKeys.keys.insert(animKey(newKeyTime, true));
                }
            }
        }
    }

    /*
        If the pre-infinity or post-infinity types aren't constant then we may
        need to add extra keyframes to simulate them.
    */
    animKeysCopy = newKeys;
    animCurveRange = animKeysCopy.getKeyRange();

    switch (animCurve.preInfinityType(&status))
    {
        case MFnAnimCurve::kConstant:
        default:
            break;
        case MFnAnimCurve::kLinear:
            break;
        case MFnAnimCurve::kCycle:
        case MFnAnimCurve::kCycleRelative:
            animCurveOffset = (newKeys.keys.begin())->time;
            while (animCurveOffset > (float)globalData->m_animStart)
            {
                newKeys.keys.insert(animKey(animCurveOffset - 0.1f, true));
                animCurveOffset -= animCurveRange;
                copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, false);
            }
            break;
        case MFnAnimCurve::kOscillate:
            animCurveOffset = (newKeys.keys.begin())->time;
            i = 0;
            while (animCurveOffset > (float)globalData->m_animStart)
            {
                newKeys.keys.insert(animKey(animCurveOffset - 0.1f, true));
                animCurveOffset -= animCurveRange;
                if (i & 0x1)
                {
                    copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, false);
                }
                else
                {
                    copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, true);
                }
                i++;
            }
            break;
    }

    switch (animCurve.postInfinityType(&status))
    {
        case MFnAnimCurve::kConstant:
        default:
            break;
        case MFnAnimCurve::kLinear:
            break;
        case MFnAnimCurve::kCycle:
        case MFnAnimCurve::kCycleRelative:
            animCurveOffset = max_element(newKeys.keys.begin(), newKeys.keys.end())->time;
            while (animCurveOffset < (float)globalData->m_animEnd)
            {
                copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, false);
                newKeys.keys.insert(animKey(animCurveOffset + 0.1f, true));
                animCurveOffset += animCurveRange;
            }
            break;
        case MFnAnimCurve::kOscillate:
            animCurveOffset = max_element(newKeys.keys.begin(), newKeys.keys.end())->time;
            i = 0;
            while (animCurveOffset < (float)globalData->m_animEnd)
            {
                if (i & 0x1)
                {
                    copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, false);
                }
                else
                {
                    copyKeyFrames(animKeysCopy, newKeys, animCurveOffset, true);
                }
                newKeys.keys.insert(animKey(animCurveOffset + 0.1f, true));
                animCurveOffset += animCurveRange;
                i++;
            }
            break;
    }

    /* Copy the animation keys we generated to out main key list */
    for (key = newKeys.keys.begin(); key != newKeys.keys.end(); key++)
    {
        float time = key->time;

        keys.insert(animKey(time, true));
    }
}

void animKeyList::addToHierarchyKeyFrames(animKeys &hierarchyKeys, unsigned int nodeIndex)
{
    animKeyIt   key;
    animKey     newKey(0.0f);

    for (key = keys.begin(); key != keys.end(); key++)
    {
        while (key->isKeyFrame == false)
        {
            key++;
        }

        if (key->time == (float)globalData->m_animStart)
        {
            newKey.time = 0.0f;
        }
        else
        {
            newKey.time = ((float)MTime(key->time, MTime::uiUnit()).as(MTime::kSeconds)) -
                            ((float)MTime(globalData->m_animStart, MTime::uiUnit()).as(MTime::kSeconds));
        }

        newKey.nodeIndex    = nodeIndex;
        newKey.matrix.pos   = key->matrix.pos;
        newKey.quat.imag    = key->quat.imag;
        newKey.quat.real    = key->quat.real;

        hierarchyKeys.insert(newKey);
    }
}

/* Convert an animKeyList to RpAnim sequences and apply them to a frame */
void animKeyList::convertToRpAnim(RwFrame *frame, RwChar *animName)
{
    RpAnimSequence  *rotSeq;
    RpAnimSequence  *transSeq;
    RwReal          keyTimeSeconds, prevKeyTimeSeconds ;
    animKeyIt       key;
    int             i, keyCount;

    keyCount = countValidKeys();

    if (keyCount > 1)
    {
        /* Create an animation sequence */
        rotSeq = RpAnimSequenceCreate(animName, rpROTATE,
                                    keyCount, keyCount - 1);

        transSeq = RpAnimSequenceCreate(animName, rpTRANSLATE,
                                        keyCount, keyCount - 1);

        /* Add the sequence to the frame */
        RpAnimFrameAddSequence(frame, rotSeq);
        RpAnimFrameAddSequence(frame, transSeq);
        
        for (i = 0, key = keys.begin(); i < keyCount; i++, key++)
        {
            while (key->isKeyFrame == false)
            {
                key++;
            }

            keyTimeSeconds =    ((float)MTime(key->time, MTime::uiUnit()).as(MTime::kSeconds)) -
                                ((float)MTime(globalData->m_animStart, MTime::uiUnit()).as(MTime::kSeconds));

            RpAnimSequenceSetRotateKey(rotSeq, i, &key->quat);
            RpAnimSequenceSetTranslateKey(transSeq, i, &key->matrix.pos);

            /* Set up the interpolators */
            if (i > 0)
            {
                /* Set the interpolators for the animation sequence */
                RpAnimSequenceSetInterpolator(rotSeq, i - 1, i - 1,
                                              i, keyTimeSeconds - prevKeyTimeSeconds);
                RpAnimSequenceSetInterpolator(transSeq, i - 1, i - 1,
                                              i, keyTimeSeconds - prevKeyTimeSeconds);
            }

            prevKeyTimeSeconds = keyTimeSeconds;
        }
    }
}

/* General animation functions */
animCurveType getAnimCurveType(MObject animCurve, MObject object)
{
    MStatus     status;
    MString     command;
    bool        result;

    MFnDagNode  objectDagNode(object, &status);
    MString     objectName = objectDagNode.name(&status);

    MFnDependencyNode   curveDepNode(animCurve, &status);
    MString             curveName = curveDepNode.name(&status);

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".tx"), &result))
    {
        if (result)
        {
            return translationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".ty"), &result))
    {
        if (result)
        {
            return translationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".tz"), &result))
    {
        if (result)
        {
            return translationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".rx"), &result))
    {
        if (result)
        {
            return rotationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".ry"), &result))
    {
        if (result)
        {
            return rotationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".rz"), &result))
    {
        if (result)
        {
            return rotationAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".sx"), &result))
    {
        if (result)
        {
            return scaleAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".sy"), &result))
    {
        if (result)
        {
            return scaleAnimCurve;
        }
    }

    if (getConnectionQueryAsBool(curveName + MString(".output"), objectName + MString(".sz"), &result))
    {
        if (result)
        {
            return scaleAnimCurve;
        }
    }

    return unknownAnimCurve;
}


void generateAnimFromHierarchyKeyFrames(animKeys &hierarchyKeys, RpSkinAnim **skinAnim,
                                        RpHAnimAnimation **hAnim)
{
    RwUInt32            i, j;
    RwInt32             numFrames = 0;
    RwInt32             frameIndex = 0;
    RwReal              early = FLT_MAX, late = -FLT_MAX;
    RwInt32             *framesLeft;
    RwReal              *lastFrame;
    RpSkinFrame         **lastSkinFramePtr = NULL;
    RpHAnimStdKeyFrame  **lastHAnimFramePtr = NULL;
    animKeyIt           key;
    RpSkinFrame         *skinFrames;
    RpHAnimStdKeyFrame  *hAnimFrames;
    unsigned int        numNodes = 0;

    /*
        Count how many different nodes are in our list. This works as all
        nodes are guaranteed to have a key on the start frame and the keylist
        is a set - no duplicates.
    */
    for (key = hierarchyKeys.begin();
        key != hierarchyKeys.end(), key->time == 0.0f;
        key++)
    {
        numNodes++;
    }

    if (numNodes == 0)
    {
        return;
    }

    framesLeft          = (RwInt32 *)malloc(sizeof(RwInt32) * numNodes);
    lastFrame           = (RwReal *)malloc(sizeof(RwReal) * numNodes);
    lastSkinFramePtr    = (RpSkinFrame **)malloc(sizeof(RpSkinFrame *) * numNodes);
    lastHAnimFramePtr   = (RpHAnimStdKeyFrame **)malloc(sizeof(RpHAnimStdKeyFrame *) * numNodes);
    
    for (i = 0; i < numNodes; i++)
    {
        framesLeft[i]           = 0;
        lastFrame[i]            = 0;
        lastSkinFramePtr[i]     = NULL;
        lastHAnimFramePtr[i]    = NULL;
    }
    
    for (key = hierarchyKeys.begin(); key != hierarchyKeys.end(); key++)
    {
        if (key->time < early)
        {
            early = key->time;
        }
        if (key->time > late)
        {
            late = key->time;
        }
        numFrames++;
        framesLeft[key->nodeIndex]++;
   }

    if (skinAnim != NULL)
    {
        *skinAnim   = RpSkinAnimCreate(numFrames, 0, late - early);
        skinFrames  = (*skinAnim)->pFrames;
    }
    
    if (hAnim != NULL)
    {
        *hAnim      = RpHAnimAnimationCreate(rpHANIMSTDKEYFRAMETYPEID, numFrames, 0, late - early);
        hAnimFrames = (RpHAnimStdKeyFrame *)((*hAnim)->pFrames);
    }

    /* grab a first and second frame for each bone */

    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < numNodes; j++)
        {
            key = hierarchyKeys.begin();

            while (key->nodeIndex != j)
            {
                key++;
            }

            if (skinAnim != NULL)
            {
                skinFrames[frameIndex].time       = key->time;
                skinFrames[frameIndex].q.imag     = key->quat.imag;
                skinFrames[frameIndex].q.real     = -key->quat.real;
                skinFrames[frameIndex].t          = key->matrix.pos;
                skinFrames[frameIndex].prevFrame  = lastSkinFramePtr[key->nodeIndex];
                lastSkinFramePtr[key->nodeIndex]  = &(skinFrames[frameIndex]);
            }

            if (hAnim != NULL)
            {
                hAnimFrames[frameIndex].time          = key->time;
                hAnimFrames[frameIndex].q.imag        = key->quat.imag;
                hAnimFrames[frameIndex].q.real        = key->quat.real;
                hAnimFrames[frameIndex].t             = key->matrix.pos;
                hAnimFrames[frameIndex].prevFrame     = lastHAnimFramePtr[key->nodeIndex];
                lastHAnimFramePtr[key->nodeIndex]     = &(hAnimFrames[frameIndex]);
            }

            lastFrame[key->nodeIndex] = key->time;

            frameIndex++;
            framesLeft[key->nodeIndex]--;

            hierarchyKeys.erase(key);
        }
    }

    /*
        Now loop through until the list is empty finding which node
        ends earliest but has frames remaining in the list.
    */
    while (hierarchyKeys.size() > 0)
    {
        RwReal lastFrameTime = FLT_MAX;
        RwInt32 lastNode = -1;

        /* find the node with the earliest last frame time */
        for (i = 0; i < numNodes; i++)
        {
            if (framesLeft[i] && lastFrame[i] < lastFrameTime)
            {
                lastFrameTime = lastFrame[i];
                lastNode = i;
            }
        }

        /*
            Find the first frame in the list for this bone,
            the list is time sorted.
        */
        for (key = hierarchyKeys.begin(); key->nodeIndex != lastNode; key++)
        {
            /* do nothing */
        }
                
        if (skinAnim != NULL)
        {
            skinFrames[frameIndex].time       = key->time;
            skinFrames[frameIndex].q.imag     = key->quat.imag;
            skinFrames[frameIndex].q.real     = -key->quat.real;
            skinFrames[frameIndex].t          = key->matrix.pos;
            skinFrames[frameIndex].prevFrame  = lastSkinFramePtr[key->nodeIndex];
            lastSkinFramePtr[key->nodeIndex]  = &(skinFrames[frameIndex]);
        }
        
        if (hAnim != NULL)
        {
            hAnimFrames[frameIndex].time          = key->time;
            hAnimFrames[frameIndex].q.imag        = key->quat.imag;
            hAnimFrames[frameIndex].q.real        = key->quat.real;
            hAnimFrames[frameIndex].t             = key->matrix.pos;
            hAnimFrames[frameIndex].prevFrame     = lastHAnimFramePtr[key->nodeIndex];
            lastHAnimFramePtr[key->nodeIndex]     = &(hAnimFrames[frameIndex]);
        }

        lastFrame[key->nodeIndex] = key->time;
        
        frameIndex++;
        framesLeft[key->nodeIndex]--;
        
        hierarchyKeys.erase(key);
    }

    free(framesLeft);
    free(lastFrame);
    free(lastSkinFramePtr);
    free(lastHAnimFramePtr);
}
